<?php
/**
 * Mark Course
 *
 * @package     AutomatorWP\Integrations\MasterStudy_LMS\Actions\Mark_Course
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_MasterStudy_LMS_Mark_Course extends AutomatorWP_Integration_Action {

    public $integration = 'masterstudy_lms';
    public $action = 'masterstudy_lms_mark_course';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_action( $this->action, array(
            'integration'       => $this->integration,
            'label'             => __( 'Mark course as completed for the user', 'automatorwp-pro' ),
            'select_option'     => __( 'Mark <strong>course as completed</strong> for the user', 'automatorwp-pro' ),
            /* translators: %1$s: Post title. */
            'edit_label'        => sprintf( __( 'Mark %1$s as completed for the user', 'automatorwp-pro' ), '{post}' ),
            /* translators: %1$s: Post title. */
            'log_label'         => sprintf( __( 'Mark %1$s as completed for the user', 'automatorwp-pro' ), '{post}' ),
            'options'           => array(
                'post' => automatorwp_utilities_post_option( array(
                    'name'              => __( 'Course:', 'automatorwp-pro' ),
                    'option_default'    => __( 'course', 'automatorwp-pro' ),
                    'option_none'       => false,
                    'option_custom'         => true,
                    'option_custom_desc'    => __( 'Course ID', 'automatorwp-pro' ),
                    'post_type'         => 'stm-courses',
                ) ),
            ),
        ) );

    }

    /**
     * Action execution function
     *
     * @since 1.0.0
     *
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     */
    public function execute( $action, $user_id, $action_options, $automation ) {

        // Shorthand
        $course_id = absint( $action_options['post'] );

        // Bail if not course provided
        if( $course_id === 0 ) {
            return;
        }

        $curriculum = get_post_meta( $course_id, 'curriculum', true );

        // Bail course curriculum is empty
        if ( empty( $curriculum ) ) {
            return;
        }

        // Turn into an array of IDs
        $curriculum = STM_LMS_Helpers::only_array_numbers( explode( ',', $curriculum ) );

        $curriculum_posts = get_posts( array(
            'post__in'       => $curriculum,
            'posts_per_page' => 999,
            'post_type'      => array( 'stm-lessons', 'stm-quizzes' ),
            'post_status'    => 'publish',
        ) );

        // Bail if no curriculum posts
        if ( empty( $curriculum_posts ) ) {
            return;
        }

        $course = stm_lms_get_user_course( $user_id, $course_id, array( 'user_course_id' ) );

        // Enroll user in this course if not enrolled yet
        if ( ! count( $course ) ) {
            STM_LMS_Course::add_user_course( $course_id, $user_id, STM_LMS_Course::item_url( $course_id, '' ), 0 );
            STM_LMS_Course::add_student( $course_id );
        }

        // Loop course items
        foreach ( $curriculum_posts as $post ) {

            if( $post->post_type === 'stm-lessons' ) {
                $lesson_id = $post->ID;

                // Skip if lesson already completed
                if ( STM_LMS_Lesson::is_lesson_completed( $user_id, $course_id, $lesson_id ) ) {
                    continue;
                }

                $end_time = time();
                $start_time = get_user_meta( $user_id, "stm_lms_course_started_{$course_id}_{$lesson_id}", true );
                $user_lesson = compact( 'user_id', 'course_id', 'lesson_id', 'start_time', 'end_time' );

                // Mark lesson as completed
                stm_lms_add_user_lesson( $user_lesson );
                STM_LMS_Course::update_course_progress( $user_id, $course_id );

                do_action( 'stm_lms_lesson_passed', $user_id, $lesson_id );

                delete_user_meta( $user_id, "stm_lms_course_started_{$course_id}_{$lesson_id}" );

            } else if( $post->post_type === 'stm-quizzes' ) {
                $quiz_id = $post->ID;

                // Skip if quiz already completed
                if ( STM_LMS_Quiz::quiz_passed( $quiz_id, $user_id ) ) {
                    continue;
                }

                $progress  = 100;
                $status    = 'passed';
                $user_quiz = compact( 'user_id', 'course_id', 'quiz_id', 'progress', 'status' );

                // Mark quiz as completed
                stm_lms_add_user_quiz( $user_quiz );
                stm_lms_get_delete_user_quiz_time( $user_id, $quiz_id );
                STM_LMS_Course::update_course_progress( $user_id, $course_id );

                $user_quiz['progress'] = round( $user_quiz['progress'], 1 );
                do_action( 'stm_lms_quiz_' . $status, $user_id, $quiz_id, $user_quiz['progress'] );
            }

        }


    }

}

new AutomatorWP_MasterStudy_LMS_Mark_Course();